<?php
// loan_history.php
require_once 'connect.php';
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if(!isset($_SESSION["logged_in"]) || $_SESSION["logged_in"] !== true) {
    header("location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];

// Handle loan payment
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if(isset($_POST['pay_emi'])) {
        $loan_id = (int)$_POST['loan_id'];
        $emi_amount = (float)$_POST['emi_amount'];
        
        // Check 15-day payment restriction
        $stmt = $conn->prepare("SELECT payment_date FROM loan_payments WHERE loan_id = ? AND user_id = ? ORDER BY payment_date DESC LIMIT 1");
        $stmt->bind_param("ii", $loan_id, $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $last_payment = $result->fetch_assoc();
        $stmt->close();
        
        if($last_payment) {
            $last_payment_date = new DateTime($last_payment['payment_date']);
            $today = new DateTime();
            $days_since_last_payment = $today->diff($last_payment_date)->days;
            
            if($days_since_last_payment < 15) {
                $_SESSION['message'] = [
                    'type' => 'warning', 
                    'text' => 'Payment Restriction',
                    'details' => 'You need to wait at least 15 days between payments. Your next payment will be available on '.$last_payment_date->modify('+15 days')->format('M d, Y'),
                    'icon' => 'fas fa-clock'
                ];
                header("Location: loan_history.php");
                exit();
            }
        }
        
        if($_SESSION['Balancer'] >= $emi_amount) {
            $conn->begin_transaction();
            try {
                // Update loan status
                $stmt = $conn->prepare("UPDATE loan_applications SET paid_emis = paid_emis + 1 WHERE id = ? AND user_id = ?");
                $stmt->bind_param("ii", $loan_id, $user_id);
                $stmt->execute();
                
                // Deduct from balance
                $new_balance = $_SESSION['Balancer'] - $emi_amount;
                $stmt = $conn->prepare("UPDATE user SET Balancer = ? WHERE id = ?");
                $stmt->bind_param("di", $new_balance, $user_id);
                $stmt->execute();
                
                // Record payment
                $stmt = $conn->prepare("INSERT INTO loan_payments (loan_id, user_id, amount, payment_date) VALUES (?, ?, ?, NOW())");
                $stmt->bind_param("iid", $loan_id, $user_id, $emi_amount);
                $stmt->execute();
                
                $conn->commit();
                $_SESSION['Balancer'] = $new_balance;
                $_SESSION['message'] = [
                    'type' => 'success', 
                    'text' => 'Payment Successful!',
                    'details' => 'Your payment of ৳'.number_format($emi_amount, 2).' has been processed.',
                    'icon' => 'fas fa-check-circle'
                ];
            } catch (Exception $e) {
                $conn->rollback();
                $_SESSION['message'] = [
                    'type' => 'danger', 
                    'text' => 'Payment Failed',
                    'details' => 'Error: '.$e->getMessage(),
                    'icon' => 'fas fa-exclamation-triangle'
                ];
            }
        } else {
            $_SESSION['message'] = [
                'type' => 'danger', 
                'text' => 'Insufficient Balance',
                'details' => 'You need ৳'.number_format($emi_amount - $_SESSION['Balancer'], 2).' more to complete this payment.',
                'icon' => 'fas fa-wallet'
            ];
            $_SESSION['redirect_to_deposit'] = true;
        }
    } elseif(isset($_POST['download_loan'])) {
        $loan_id = (int)$_POST['loan_id'];
        
        // Fetch loan details
        $stmt = $conn->prepare("SELECT 
                                la.id as loan_app_id,
                                l.loanName, 
                                la.amount, 
                                la.duration, 
                                la.status, 
                                la.created_at,
                                la.paid_emis,
                                p.loanTime,
                                p.profitPersatice,
                                p.totalEmi,
                                p.totalEmiAmount,
                                p.TotalPayableAmount
                               FROM loan_applications la 
                               JOIN loanname l ON la.loan_id = l.id 
                               JOIN productloan p ON la.loan_id = p.loanId AND la.duration = p.loanTime
                               WHERE la.id = ? AND la.user_id = ?");
        $stmt->bind_param("ii", $loan_id, $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $loan = $result->fetch_assoc();
        $stmt->close();
        
        if($loan) {
            // Create image with loan statement
            $width = 800;
            $height = 1000;
            $image = imagecreatetruecolor($width, $height);
            
            // Colors
            $white = imagecolorallocate($image, 255, 255, 255);
            $black = imagecolorallocate($image, 0, 0, 0);
            $primary = imagecolorallocate($image, 36, 117, 201);
            $gray = imagecolorallocate($image, 100, 100, 100);
            $lightGray = imagecolorallocate($image, 200, 200, 200);
            
            // Fill background
            imagefilledrectangle($image, 0, 0, $width, $height, $white);
            
            // Font handling - try multiple possible font paths
            $font = null;
            $fontPaths = [
                'arial.ttf',
                '/usr/share/fonts/truetype/arial.ttf',
                '/usr/share/fonts/truetype/msttcorefonts/arial.ttf',
                '/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf', // Fallback font
                '/usr/share/fonts/truetype/liberation/LiberationSans-Regular.ttf' // Another fallback
            ];
            
            foreach ($fontPaths as $fontPath) {
                if (file_exists($fontPath)) {
                    $font = $fontPath;
                    break;
                }
            }
            
            if (!$font) {
                // If no font found, use default (but text won't be as nice)
                $font = 5; // Built-in GD font
            }
            
            // Function to handle text output with fallback
            function safeImageText($image, $size, $angle, $x, $y, $color, $font, $text) {
                if (is_int($font)) {
                    // Use built-in GD font
                    imagestring($image, $font, $x, $y, $text, $color);
                } else {
                    // Use TrueType font
                    imagettftext($image, $size, $angle, $x, $y, $color, $font, $text);
                }
            }
            
            // Add header
            safeImageText($image, 20, 0, 50, 50, $primary, $font, "Easy Loan Statement");
            safeImageText($image, 16, 0, 50, 80, $gray, $font, "Loan ID: #".$loan['loan_app_id']);
            safeImageText($image, 16, 0, 50, 110, $gray, $font, "Date: ".date('M d, Y', strtotime($loan['created_at'])));
            
            // Add loan details
            $y = 160;
            safeImageText($image, 18, 0, 50, $y, $primary, $font, "Loan Details");
            $y += 40;
            
            safeImageText($image, 14, 0, 50, $y, $black, $font, "Loan Type:");
            safeImageText($image, 14, 0, 250, $y, $black, $font, $loan['loanName']);
            $y += 30;
            
            safeImageText($image, 14, 0, 50, $y, $black, $font, "Loan Amount:");
            safeImageText($image, 14, 0, 250, $y, $black, $font, "৳".number_format($loan['amount'], 2));
            $y += 30;
            
            safeImageText($image, 14, 0, 50, $y, $black, $font, "Duration:");
            safeImageText($image, 14, 0, 250, $y, $black, $font, $loan['duration']." months");
            $y += 30;
            
            safeImageText($image, 14, 0, 50, $y, $black, $font, "Interest Rate:");
            safeImageText($image, 14, 0, 250, $y, $black, $font, $loan['profitPersatice']."%");
            $y += 30;
            
            safeImageText($image, 14, 0, 50, $y, $black, $font, "EMI Amount:");
            safeImageText($image, 14, 0, 250, $y, $black, $font, "৳".number_format($loan['totalEmiAmount'], 2));
            $y += 30;
            
            safeImageText($image, 14, 0, 50, $y, $black, $font, "Total Payable:");
            safeImageText($image, 14, 0, 250, $y, $black, $font, "৳".number_format($loan['TotalPayableAmount'], 2));
            $y += 40;
            
            // Add payment schedule
            safeImageText($image, 18, 0, 50, $y, $primary, $font, "Payment Schedule");
            $y += 40;
            
            // Table header
            safeImageText($image, 14, 0, 50, $y, $black, $font, "Month");
            safeImageText($image, 14, 0, 150, $y, $black, $font, "Due Date");
            safeImageText($image, 14, 0, 350, $y, $black, $font, "Amount");
            safeImageText($image, 14, 0, 500, $y, $black, $font, "Status");
            $y += 5;
            imageline($image, 50, $y, 750, $y, $lightGray);
            $y += 25;
            
            // Add payment rows
            for($i=1; $i<=$loan['totalEmi']; $i++) {
                $due_date = date('M d, Y', strtotime($loan['created_at'] . " + $i months"));
                $is_paid = $i <= $loan['paid_emis'];
                
                safeImageText($image, 12, 0, 50, $y, $black, $font, "Month $i");
                safeImageText($image, 12, 0, 150, $y, $black, $font, $due_date);
                safeImageText($image, 12, 0, 350, $y, $black, $font, "৳".number_format($loan['totalEmiAmount'], 2));
                
                if($is_paid) {
                    safeImageText($image, 12, 0, 500, $y, $black, $font, "Paid");
                } else {
                    safeImageText($image, 12, 0, 500, $y, $black, $font, "Pending");
                }
                
                $y += 30;
            }
            
            // Add footer
            $y += 20;
            safeImageText($image, 12, 0, 50, $y, $gray, $font, "Generated on: ".date('M d, Y H:i:s'));
            
            // Clear any output buffers to prevent header errors
            while (ob_get_level()) {
                ob_end_clean();
            }
            
            // Output the image
            header('Content-Type: image/png');
            header('Content-Disposition: attachment; filename="loan_statement_'.$loan['loan_app_id'].'.png"');
            imagepng($image);
            imagedestroy($image);
            exit();
        }
    }
    
    header("Location: loan_history.php");
    exit();
}

// Fetch user balance
$stmt = $conn->prepare("SELECT Balancer FROM user WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$stmt->close();

// Fetch loan history
$loan_history = [];
$stmt = $conn->prepare("SELECT 
                        la.id as loan_app_id,
                        l.loanName, 
                        la.amount, 
                        la.duration, 
                        la.advance,
                        la.status, 
                        la.created_at,
                        la.paid_emis,
                        p.loanTime,
                        p.profitPersatice,
                        p.totalEmi,
                        p.totalEmiAmount,
                        p.TotalPayableAmount
                       FROM loan_applications la 
                       JOIN loanname l ON la.loan_id = l.id 
                       JOIN productloan p ON la.loan_id = p.loanId AND la.duration = p.loanTime
                       WHERE la.user_id = ? 
                       ORDER BY la.created_at DESC");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$loan_history = $result->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch payments for each loan
foreach($loan_history as &$loan) {
    $stmt = $conn->prepare("SELECT * FROM loan_payments WHERE loan_id = ? AND user_id = ? ORDER BY payment_date DESC");
    $stmt->bind_param("ii", $loan['loan_app_id'], $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $loan['payments'] = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
    
    $created_date = new DateTime($loan['created_at']);
    $loan['next_payment_date'] = $created_date->modify('+'.($loan['paid_emis'] + 1).' months')->format('Y-m-d');
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Loan History - Easy Loan</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-color: #2475c9;
            --secondary-color: #F6BE00;
            --success-color: #28a745;
            --warning-color: #ffc107;
            --danger-color: #dc3545;
            --light-bg: #f8f9fa;
            --card-shadow: 0 4px 12px rgba(0,0,0,0.08);
        }
        
        * {
            font-family: 'Poppins', sans-serif;
            -webkit-user-select: none;
            -moz-user-select: none;
            user-select: none;
        }
        
        body {
            background-color: var(--light-bg);
            padding-bottom: 20px;
        }
        
        .header {
            height: 60px;
            background: linear-gradient(135deg, var(--primary-color), #1a5f9c);
            display: flex;
            align-items: center;
            padding: 0 20px;
            color: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .header-title {
            font-size: 1.3rem;
            font-weight: 600;
            margin-left: 15px;
            flex-grow: 1;
            text-align: center;
        }
        
        .back-btn {
            color: white;
            font-size: 1.4rem;
            text-decoration: none;
        }
        
        .main-container {
            max-width: 800px;
            margin: 0 auto;
            padding: 20px 15px;
        }
        
        .custom-alert {
            border-radius: 10px;
            border-left: 5px solid;
            padding: 15px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            box-shadow: var(--card-shadow);
        }
        
        .alert-success {
            background-color: rgba(40, 167, 69, 0.15);
            border-color: var(--success-color);
        }
        
        .alert-warning {
            background-color: rgba(255, 193, 7, 0.15);
            border-color: var(--warning-color);
        }
        
        .alert-danger {
            background-color: rgba(220, 53, 69, 0.15);
            border-color: var(--danger-color);
        }
        
        .loan-card {
            background: white;
            border-radius: 12px;
            box-shadow: var(--card-shadow);
            margin-bottom: 25px;
            overflow: hidden;
        }
        
        .loan-header {
            padding: 18px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
        }
        
        .loan-amount {
            font-size: 1.4rem;
            font-weight: 700;
            color: var(--primary-color);
        }
        
        .status-badge {
            padding: 6px 15px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        
        .status-approved { background-color: var(--success-color); color: white; }
        
        .loan-details {
            padding: 20px;
        }
        
        .info-item {
            display: flex;
            justify-content: space-between;
            margin-bottom: 15px;
        }
        
        .payment-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0;
            margin-top: 15px;
            border-radius: 10px;
            overflow: hidden;
        }
        
        .payment-table th {
            background-color: var(--primary-color);
            color: white;
            padding: 12px 15px;
            text-align: left;
        }
        
        .payment-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #f0f0f0;
        }
        
        .btn-pay {
            background-color: var(--primary-color);
            color: white;
            border: none;
            border-radius: 6px;
            padding: 8px 16px;
            font-size: 0.8rem;
        }
        
        .btn-download {
            background-color: var(--success-color);
            color: white;
            border: none;
            border-radius: 6px;
            padding: 10px 20px;
            width: 100%;
            margin-top: 15px;
        }
        
        @media (max-width: 576px) {
            .header-title {
                font-size: 1.1rem;
            }
            
            .loan-amount {
                font-size: 1.2rem;
            }
            
            .payment-table th, 
            .payment-table td {
                padding: 8px 10px;
                font-size: 0.75rem;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <a href="/" class="back-btn">
            <i class="fas fa-arrow-left"></i>
        </a>
        <div class="header-title">Loan History</div>
        <div style="width: 24px;"></div>
    </div>

    <div class="main-container">
        <?php if(isset($_SESSION['message'])): ?>
            <div class="custom-alert alert-<?= $_SESSION['message']['type'] ?>">
                <i class="<?= $_SESSION['message']['icon'] ?> me-3"></i>
                <div>
                    <strong><?= $_SESSION['message']['text'] ?></strong>
                    <div><?= $_SESSION['message']['details'] ?></div>
                </div>
            </div>
            <?php unset($_SESSION['message']); ?>
        <?php endif; ?>
        
        <?php if(empty($loan_history)): ?>
            <div class="custom-alert alert-info">
                <i class="fas fa-info-circle me-3"></i>
                <div>You haven't applied for any loans yet.</div>
            </div>
        <?php else: ?>
            <?php foreach($loan_history as $loan): ?>
                <div class="loan-card">
                    <div class="loan-header">
                        <div>
                            <div class="loan-amount">৳<?= number_format($loan['amount'], 2) ?></div>
                            <div class="loan-term"><?= $loan['loanName'] ?> • <?= $loan['duration'] ?> months</div>
                        </div>
                        <span class="status-badge status-<?= $loan['status'] ?>">
                            <?= ucfirst($loan['status']) ?>
                        </span>
                    </div>
                    
                    <div class="loan-details">
                        <div class="info-item">
                            <span>Interest Rate</span>
                            <span><?= $loan['profitPersatice'] ?>%</span>
                        </div>
                        <div class="info-item">
                            <span>EMI Amount</span>
                            <span>৳<?= number_format($loan['totalEmiAmount'], 2) ?>/month</span>
                        </div>
                        <div class="info-item">
                            <span>Total Payable</span>
                            <span>৳<?= number_format($loan['TotalPayableAmount'], 2) ?></span>
                        </div>
                        <div class="info-item">
                            <span>Next Payment</span>
                            <span><?= date('M d, Y', strtotime($loan['next_payment_date'])) ?></span>
                        </div>
                        
                        <form method="post">
                            <input type="hidden" name="loan_id" value="<?= $loan['loan_app_id'] ?>">
                            <button type="submit" name="download_loan" class="btn-download">
                                <i class="fas fa-file-image me-2"></i>Download Statement (PNG)
                            </button>
                        </form>
                    </div>
                    
                    <div style="padding: 0 20px 20px;">
                        <h6 style="font-weight: 600; margin-bottom: 15px; color: var(--primary-color);">
                            <i class="fas fa-calendar-alt me-2"></i>Payment Schedule
                        </h6>
                        <table class="payment-table">
                            <thead>
                                <tr>
                                    <th>Month</th>
                                    <th>Due Date</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php for($i=1; $i<=$loan['totalEmi']; $i++): ?>
                                    <?php 
                                        $due_date = date('M d, Y', strtotime($loan['created_at'] . " + $i months"));
                                        $is_paid = $i <= $loan['paid_emis'];
                                        $is_current = $i == ($loan['paid_emis'] + 1);
                                    ?>
                                    <tr class="<?= $is_paid ? 'paid-emi' : ($is_current ? 'current-emi' : '') ?>">
                                        <td>Month <?= $i ?></td>
                                        <td><?= $due_date ?></td>
                                        <td>৳<?= number_format($loan['totalEmiAmount'], 2) ?></td>
                                        <td>
                                            <?php if($is_paid): ?>
                                                <span class="text-success"><i class="fas fa-check-circle me-1"></i> Paid</span>
                                            <?php elseif($is_current): ?>
                                                <span class="text-warning"><i class="fas fa-exclamation-circle me-1"></i> Due</span>
                                            <?php else: ?>
                                                <span class="text-muted"><i class="fas fa-clock me-1"></i> Pending</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if($is_current && $loan['status'] == 'approved'): ?>
                                                <form method="post" style="display:inline;">
                                                    <input type="hidden" name="loan_id" value="<?= $loan['loan_app_id'] ?>">
                                                    <input type="hidden" name="emi_amount" value="<?= $loan['totalEmiAmount'] ?>">
                                                    <button type="submit" name="pay_emi" class="btn-pay">
                                                        <i class="fas fa-money-bill-wave me-1"></i> Pay
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endfor; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
  <style>
  body {
      margin: 0;
      padding: 0;
      overflow-x: hidden;
    }

    .footer {
      position: fixed;
      bottom: 0;
      width: 100%;
      background: rgba(15, 32, 39, 0.95);
      backdrop-filter: blur(4px);
      color: white;
      padding: 10px 0;
      box-shadow: 0 -4px 20px rgba(0, 0, 0, 0.4);
      z-index: 999;
      transition: background 0.3s ease;
    }

    .footer.scrolled {
      background: rgba(15, 32, 39, 0.8);
    }

    .footer .row {
      display: flex;
      justify-content: space-around;
      align-items: center;
      text-align: center;
      max-width: 1000px;
      margin: 0 auto;
      padding: 0 10px;
      flex-wrap: wrap;
      box-sizing: border-box;
      overflow-x: hidden;
    }

    .footer .col {
      flex: 1 1 22%;
      min-width: 60px;
      max-width: 25%;
      margin: 5px 0;
      box-sizing: border-box;
    }

    .footer .col a {
      display: flex;
      flex-direction: column;
      align-items: center;
      text-decoration: none;
      color: white;
      transition: transform 0.3s;
    }

    .footer .col a:hover {
      transform: scale(1.1);
    }

    .footer-icon {
      width: 100%;
      max-width: 48px;
      height: auto;
      object-fit: contain;
      transition: all 0.3s ease-in-out;
      filter: drop-shadow(0 4px 8px rgba(0,0,0,0.4));
    }

    .footer p {
      margin: 4px 0 0;
      font-size: 14px;
    }

    .footer .col:nth-child(1) img {
      animation: float1 4s ease-in-out infinite;
      filter: drop-shadow(0 5px 12px rgba(255, 111, 0, 0.4));
    }

    .footer .col:nth-child(2) img {
      animation: float2 5s ease-in-out infinite;
      filter: drop-shadow(0 5px 12px rgba(37, 211, 102, 0.5));
    }

    .footer .col:nth-child(3) img {
      animation: float3 4.5s ease-in-out infinite;
      filter: drop-shadow(0 5px 12px rgba(255, 255, 0, 0.5));
    }

    .footer .col:nth-child(4) img {
      animation: float4 5.2s ease-in-out infinite;
      filter: drop-shadow(0 5px 12px rgba(0, 123, 255, 0.5));
    }

    @keyframes float1 {
      0%, 100% { transform: translateY(0px) rotateZ(0deg); }
      50% { transform: translateY(-4px) rotateZ(1deg); }
    }

    @keyframes float2 {
      0%, 100% { transform: translateY(0px) rotateZ(0deg); }
      50% { transform: translateY(-3px) rotateZ(-1deg); }
    }

    @keyframes float3 {
      0%, 100% { transform: translateY(0px) rotateZ(0deg); }
      50% { transform: translateY(-5px) rotateZ(1deg); }
    }

    @keyframes float4 {
      0%, 100% { transform: translateY(0px) rotateZ(0deg); }
      50% { transform: translateY(-3px) rotateZ(-1.5deg); }
    }

    @media (max-width: 768px) {
      .footer-icon {
        max-width: 38px;
      }
      .footer p {
        font-size: 12px;
      }
    }

    @media (max-width: 480px) {
      .footer .row {
        flex-direction: row;
        justify-content: space-between;
        padding: 0 10px;
      }
      .footer-icon {
        max-width: 32px;
      }
      .footer p {
        font-size: 11px;
      }
      .footer .col {
        flex: 1 1 22%;
      }
    }
  </style>
</head>
<body>
 <!-- Footer Navigation -->
<div class="footer">
    <div class="row">
        <div class="col">
            <a href="dashboard.php">
                <img class="footer-icon" src="images/footer/home.png" alt="Home">
                <p>Home</p>
            </a>
        </div>
        <div class="col">
            <a href="https://wa.me/<?= $whatsapp_number ?>" target="_blank">
                <img class="footer-icon" src="images/footer/whatsapp.png" alt="WhatsApp">
                <p>WhatsApp</p>
            </a>
        </div>
        <div class="col">
            <a href="profile.php">
                <img class="footer-icon" src="images/footer/user-4.png" alt="Profile">
                <p>Profile</p>
            </a>
        </div>
        <div class="col">
            <a href="<?= $_SESSION["logged_in"] ? 'logout.php' : 'login.php' ?>">
                <img class="footer-icon" src="images/footer/<?= $_SESSION["logged_in"] ? 'logout-2.png' : 'login.png' ?>" alt="Login">
                <p><?= $_SESSION["logged_in"] ? 'LogOut' : 'Login' ?></p>
            </a>
        </div>
    </div>
</div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>